(() => {
  // js/config.js
  var Config = {
    // API Configuration
    API_URL: "https://translate-api-70536743886.europe-west4.run.app/translate",
    // Cache Configuration
    CACHE_MAX_SIZE: 200,
    CACHE_STORAGE_KEY: "translationCache",
    // Timing Constants (in milliseconds)
    TIMING: {
      INIT_DELAY: 500,
      PAGE_LOAD_CHECK: 1e3,
      API_TIMEOUT: 5e3,
      STATUS_DISPLAY: 5e3,
      MAX_INIT_ATTEMPTS: 3
    },
    // DOM Selectors
    SELECTORS: {
      // Chat input selectors
      MAIN_CHAT_INPUT: '.theatermodeInputFieldChat[contenteditable="true"]',
      PM_CHAT_INPUT: '.theatermodeInputFieldPm[contenteditable="true"]',
      CHAT_INPUT_FIELD: '.chat-input-field[contenteditable="true"]',
      CHAT_INPUT_TESTID: '[data-testid="chat-input"][contenteditable="true"]',
      // Chat message selectors
      CHAT_MESSAGE: '[data-testid="chat-message"]',
      CHAT_MESSAGE_TEXT: '[data-testid="chat-message-text"]',
      MSG_LIST_WRAPPER: ".msg-list-wrapper-split"
    },
    // Theme Colors
    COLORS: {
      // Backgrounds
      BG_PRIMARY: "rgba(17, 24, 39, 0.95)",
      BG_INPUT: "rgba(255, 255, 255, 0.1)",
      BG_INPUT_FOCUS: "rgba(255, 255, 255, 0.15)",
      BG_BUTTON: "rgba(255, 255, 255, 0.05)",
      BG_BUTTON_HOVER: "rgba(255, 255, 255, 0.15)",
      // Borders
      BORDER_DEFAULT: "rgba(255, 255, 255, 0.3)",
      BORDER_FOCUS: "rgba(255, 255, 255, 0.5)",
      BORDER_BUTTON: "rgba(255, 255, 255, 0.4)",
      BORDER_BUTTON_HOVER: "rgba(255, 255, 255, 0.6)",
      // Text
      TEXT_PRIMARY: "#ffffff",
      TEXT_SECONDARY: "rgba(255, 255, 255, 0.7)",
      TEXT_PLACEHOLDER: "rgba(255, 255, 255, 0.4)",
      // Status Colors
      STATUS_SUCCESS: "rgba(100, 255, 150, 0.9)",
      STATUS_ERROR: "rgba(255, 100, 100, 0.9)",
      STATUS_WARNING: "rgba(255, 200, 100, 0.9)",
      STATUS_INFO: "rgba(255, 255, 255, 0.6)",
      // Translation element colors
      TRANSLATION_BG: "rgb(22, 30, 42)",
      TRANSLATION_TEXT: "rgb(209, 233, 239)",
      TRANSLATION_BORDER: "rgb(79, 145, 162)"
    },
    // Default Language Settings
    DEFAULTS: {
      ROOM_LANGUAGE: "en",
      MY_LANGUAGE: "ru"
    },
    // UI Text
    TEXT: {
      TRANSLATE_BTN: "\u041F\u0435\u0440\u0435\u0432\u0435\u0441\u0442\u0438",
      TRANSLATING: "\u041F\u0435\u0440\u0435\u0432\u043E\u0436\u0443...",
      CONFIGURED: "\u041D\u0430\u0441\u0442\u0440\u043E\u0435\u043D",
      NOT_CONFIGURED: "\u041D\u0435 \u043D\u0430\u0441\u0442\u0440\u043E\u0435\u043D",
      NOT_SET: "\u041D\u0435 \u0443\u0441\u0442\u0430\u043D\u043E\u0432\u043B\u0435\u043D",
      ACTIVE: "\u0410\u043A\u0442\u0438\u0432\u043D\u0430",
      INACTIVE: "\u041D\u0435\u0430\u043A\u0442\u0438\u0432\u043D\u0430",
      UNKNOWN: "\u041D\u0435\u0438\u0437\u0432\u0435\u0441\u0442\u043D\u043E",
      ERROR: "\u041E\u0448\u0438\u0431\u043A\u0430",
      // Error messages
      ERR_ENTER_TEXT: "Please enter some text to translate",
      ERR_CONFIGURE_KEY: "Please configure your API key in extension options",
      ERR_NO_CHAT_INPUT: "Error: Could not find chat input field",
      ERR_API_KEY_REQUIRED: "\u041F\u043E\u0436\u0430\u043B\u0443\u0439\u0441\u0442\u0430, \u0432\u0432\u0435\u0434\u0438\u0442\u0435 API \u043A\u043B\u044E\u0447",
      ERR_INVALID_API_KEY: "Invalid API key. Please check your API key in extension options.",
      ERR_API_FORBIDDEN: "API key is not authorized to use this service.",
      ERR_NO_API_KEY: "API key not configured. Please set it in extension options.",
      ERR_INVALID_RESPONSE: "Invalid API response format",
      // Success messages
      SUCCESS_SETTINGS_SAVED: "\u041D\u0430\u0441\u0442\u0440\u043E\u0439\u043A\u0438 \u0441\u043E\u0445\u0440\u0430\u043D\u0435\u043D\u044B \u0443\u0441\u043F\u0435\u0448\u043D\u043E!",
      SUCCESS_API_KEY_CLEARED: "API \u043A\u043B\u044E\u0447 \u043E\u0447\u0438\u0449\u0435\u043D",
      // Info messages
      INFO_TESTING_KEY: "\u041F\u0440\u043E\u0432\u0435\u0440\u043A\u0430 API \u043A\u043B\u044E\u0447\u0430 \u0438 \u0441\u043E\u0435\u0434\u0438\u043D\u0435\u043D\u0438\u044F...",
      INFO_EXTENSION_READY: "\u0420\u0430\u0441\u0448\u0438\u0440\u0435\u043D\u0438\u0435 \u0433\u043E\u0442\u043E\u0432\u043E! \u0418\u043D\u0442\u0435\u0440\u0444\u0435\u0439\u0441 \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430 \u043F\u043E\u044F\u0432\u0438\u0442\u0441\u044F, \u0435\u0441\u043B\u0438 \u043D\u0430 \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u043E \u043F\u043E\u043B\u0435 \u0432\u0432\u043E\u0434\u0430 \u0447\u0430\u0442\u0430.",
      INFO_CONFIGURE_KEY: '\u041F\u043E\u0436\u0430\u043B\u0443\u0439\u0441\u0442\u0430, \u043D\u0430\u0441\u0442\u0440\u043E\u0439\u0442\u0435 \u0432\u0430\u0448 API \u043A\u043B\u044E\u0447 \u0434\u043B\u044F \u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u044F \u0441\u0435\u0440\u0432\u0438\u0441\u0430 \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430. \u041D\u0430\u0436\u043C\u0438\u0442\u0435 "\u041D\u0430\u0441\u0442\u0440\u043E\u0438\u0442\u044C API \u043A\u043B\u044E\u0447" \u043D\u0438\u0436\u0435.',
      // Placeholders
      PLACEHOLDER_INPUT: "\u0412\u0432\u0435\u0434\u0438\u0442\u0435 \u0440\u0443\u0441\u0441\u043A\u0438\u0439 \u0442\u0435\u043A\u0441\u0442 \u0434\u043B\u044F \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430..."
    }
  };

  // js/utils.js
  var Utils = {
    /**
     * Generate a cache key for translations
     * @param {string} text - Text to translate
     * @param {string} sourceLang - Source language code
     * @param {string} targetLang - Target language code
     * @returns {string}
     */
    generateCacheKey(text, sourceLang, targetLang) {
      return `${sourceLang}:${targetLang}:${text}`;
    },
    /**
     * Initialize or retrieve cache data
     * @param {Object} data - Storage data object
     * @returns {Object} Cache object with translations and order
     */
    initializeCache(data) {
      return data[Config.CACHE_STORAGE_KEY] || { translations: {}, order: [] };
    },
    /**
     * Create a mutation observer for added nodes
     * @param {Function} handler - Function to call for each added element node
     * @param {Object} options - MutationObserver options
     * @returns {MutationObserver}
     */
    createNodeObserver(handler, options = { childList: true, subtree: true }) {
      const observer = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
          if (!mutation.addedNodes.length) continue;
          for (const node of mutation.addedNodes) {
            if (node.nodeType === Node.ELEMENT_NODE) {
              handler(node);
            }
          }
        }
      });
      observer.observe(document.body, options);
      return observer;
    },
    /**
     * Add focus/blur effect handlers to an element
     * @param {HTMLElement} element - Element to add effects to
     * @param {Object} focusStyles - Styles to apply on focus
     * @param {Object} blurStyles - Styles to apply on blur
     */
    addFocusEffects(element, focusStyles, blurStyles) {
      element.addEventListener("focus", () => {
        Object.assign(element.style, focusStyles);
      });
      element.addEventListener("blur", () => {
        Object.assign(element.style, blurStyles);
      });
    },
    /**
     * Add hover effect handlers to an element
     * @param {HTMLElement} element - Element to add effects to
     * @param {Object} hoverStyles - Styles to apply on hover
     * @param {Object} defaultStyles - Styles to apply when not hovering
     */
    addHoverEffects(element, hoverStyles, defaultStyles) {
      element.addEventListener("mouseenter", () => {
        if (!element.disabled) {
          Object.assign(element.style, hoverStyles);
        }
      });
      element.addEventListener("mouseleave", () => {
        if (!element.disabled) {
          Object.assign(element.style, defaultStyles);
        }
      });
    },
    /**
     * Set status message with appropriate styling
     * @param {HTMLElement} element - Status element
     * @param {string} message - Message to display
     * @param {'success'|'error'|'warning'|'info'} type - Status type
     */
    setStatus(element, message, type) {
      const colorMap = {
        success: Config.COLORS.STATUS_SUCCESS,
        error: Config.COLORS.STATUS_ERROR,
        warning: Config.COLORS.STATUS_WARNING,
        info: Config.COLORS.STATUS_INFO
      };
      element.textContent = message;
      element.style.color = colorMap[type] || colorMap.info;
    },
    /**
     * Show a status message that auto-hides (for options/popup pages)
     * @param {HTMLElement} element - Status element
     * @param {string} message - Message to display
     * @param {'success'|'error'|'info'} type - Status type
     * @param {number} duration - Duration in ms before hiding
     */
    showStatusMessage(element, message, type, duration) {
      element.textContent = message;
      element.className = `status ${type} show`;
      setTimeout(() => {
        element.className = "status";
      }, duration || Config.TIMING.STATUS_DISPLAY);
    },
    /**
     * Make a translation API request
     * @param {string} apiUrl - API endpoint URL
     * @param {string} apiKey - API key
     * @param {string} text - Text to translate
     * @param {string} source - Source language code
     * @param {string} target - Target language code
     * @param {number} timeout - Request timeout in ms
     * @returns {Promise<{success: boolean, translatedText?: string, error?: string}>}
     */
    async fetchTranslation(apiUrl, apiKey, text, source, target, timeout) {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), timeout || Config.TIMING.API_TIMEOUT);
      try {
        const response = await fetch(apiUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            "X-API-Key": apiKey
          },
          body: JSON.stringify({ text, source, target }),
          signal: controller.signal
        });
        clearTimeout(timeoutId);
        if (!response.ok) {
          const errorData = await response.json().catch(() => ({}));
          throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
        }
        const data = await response.json();
        if (data?.data?.translations?.[0]?.translatedText) {
          return {
            success: true,
            translatedText: data.data.translations[0].translatedText
          };
        } else {
          throw new Error("Invalid response format from translation API");
        }
      } catch (error) {
        clearTimeout(timeoutId);
        if (error.name === "AbortError") {
          return { success: false, error: `Connection timeout (${(timeout || Config.TIMING.API_TIMEOUT) / 1e3}s)` };
        } else if (error.message.includes("Failed to fetch") || error.message.includes("NetworkError")) {
          return { success: false, error: "Cannot connect to server - check network connection" };
        } else {
          return { success: false, error: error.message };
        }
      }
    },
    /**
     * Get language settings from storage
     * @returns {Promise<{myLanguage: string, roomLanguage: string}>}
     */
    async getLanguageSettings() {
      const config = await chrome.storage.sync.get(["myLanguage", "roomLanguage"]);
      return {
        myLanguage: config.myLanguage || Config.DEFAULTS.MY_LANGUAGE,
        roomLanguage: config.roomLanguage || Config.DEFAULTS.ROOM_LANGUAGE
      };
    },
    /**
     * Log a message with the extension prefix
     * @param {string} context - Context identifier (e.g., 'BG', 'Content')
     * @param {string} message - Message to log
     * @param {...any} args - Additional arguments
     */
    log(context, message, ...args) {
      console.log(`[Chat Translate ${context}]`, message, ...args);
    },
    /**
     * Log an error with the extension prefix
     * @param {string} context - Context identifier
     * @param {string} message - Error message
     * @param {...any} args - Additional arguments
     */
    logError(context, message, ...args) {
      console.error(`[Chat Translate ${context}]`, message, ...args);
    }
  };

  // content.js
  var C = Config;
  var U = Utils;
  var hasApiKey = false;
  U.log("Content", `v${chrome.runtime.getManifest().version} loaded`);
  var backgroundReady = false;
  var backgroundReadyPromise = new Promise((resolve) => {
    chrome.runtime.sendMessage({ type: "PING" }, () => {
      if (chrome.runtime.lastError) {
        console.warn("[Chat Translate] Background ping failed:", chrome.runtime.lastError.message);
      }
      backgroundReady = true;
      resolve();
    });
  });
  async function checkApiKeyStatus() {
    try {
      const response = await chrome.runtime.sendMessage({ type: "CHECK_API_KEY" });
      hasApiKey = response?.hasKey || false;
      return hasApiKey;
    } catch (error) {
      U.logError("Content", "Failed to check API key:", error);
      hasApiKey = false;
      return false;
    }
  }
  var CACHE_LOCK_NAME = "chat-translate-cache";
  async function getCachedTranslation(cacheKey) {
    try {
      await backgroundReadyPromise;
      const data = await chrome.storage.session.get(C.CACHE_STORAGE_KEY);
      const cache = U.initializeCache(data);
      return cache.translations[cacheKey] || null;
    } catch (error) {
      U.logError("Content", "Cache read error:", error);
      return null;
    }
  }
  async function setCachedTranslation(cacheKey, translatedText) {
    try {
      await backgroundReadyPromise;
      await navigator.locks.request(CACHE_LOCK_NAME, async () => {
        const data = await chrome.storage.session.get(C.CACHE_STORAGE_KEY);
        const cache = U.initializeCache(data);
        if (!cache.translations[cacheKey]) {
          cache.order.push(cacheKey);
          while (cache.order.length > C.CACHE_MAX_SIZE) {
            const oldestKey = cache.order.shift();
            delete cache.translations[oldestKey];
          }
        }
        cache.translations[cacheKey] = translatedText;
        await chrome.storage.session.set({ [C.CACHE_STORAGE_KEY]: cache });
      });
    } catch (error) {
      U.logError("Content", "Cache write error:", error);
    }
  }
  async function translateText(text, sourceLang, targetLang) {
    const cacheKey = U.generateCacheKey(text, sourceLang, targetLang);
    const cachedResult = await getCachedTranslation(cacheKey);
    if (cachedResult) {
      U.log("Content", "Cache hit");
      return { success: true, translatedText: cachedResult };
    }
    return new Promise((resolve) => {
      chrome.runtime.sendMessage(
        { type: "TRANSLATE", text, source: sourceLang, target: targetLang },
        async (response) => {
          if (chrome.runtime.lastError) {
            U.logError("Content", "Runtime error:", chrome.runtime.lastError.message);
            resolve({ success: false, error: chrome.runtime.lastError.message });
            return;
          }
          if (response.success && response.translatedText) {
            await setCachedTranslation(cacheKey, response.translatedText);
          }
          resolve(response);
        }
      );
    });
  }
  function findChatInputForms() {
    const forms = [];
    const mainChatInput = document.querySelector(C.SELECTORS.MAIN_CHAT_INPUT);
    if (mainChatInput) {
      const form = mainChatInput.closest("form");
      if (form) forms.push({ form, type: "main" });
    }
    const pmChatInputs = document.querySelectorAll(C.SELECTORS.PM_CHAT_INPUT);
    pmChatInputs.forEach((pmChatInput, index) => {
      const pmForm = pmChatInput.closest("form");
      if (pmForm) forms.push({ form: pmForm, type: `pm-${index}` });
    });
    return forms;
  }
  function findChatInput(form) {
    return form.querySelector(C.SELECTORS.CHAT_INPUT_FIELD) || form.querySelector(C.SELECTORS.CHAT_INPUT_TESTID);
  }
  function createSetupNotice(formDiv, type = "main") {
    const containerId = `chat-translate-container-${type}`;
    if (document.getElementById(containerId)) return;
    const container = document.createElement("div");
    container.id = containerId;
    container.style.cssText = `
        margin-top: 10px;
        padding: 12px 16px;
        background: ${C.COLORS.BG_PRIMARY};
        backdrop-filter: blur(20px);
        -webkit-backdrop-filter: blur(20px);
        border-radius: 6px;
        box-shadow: 0 8px 32px 0 rgba(0, 0, 0, 0.4);
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 12px;
    `;
    const messageDiv = document.createElement("div");
    messageDiv.style.cssText = `
        display: flex;
        align-items: center;
        gap: 10px;
        color: ${C.COLORS.STATUS_WARNING};
        font-size: 13px;
    `;
    messageDiv.innerHTML = `
        <span style="font-size: 18px;">\u26A0\uFE0F</span>
        <span>\u041F\u0435\u0440\u0435\u0432\u043E\u0434\u0447\u0438\u043A \u043D\u0435 \u043D\u0430\u0441\u0442\u0440\u043E\u0435\u043D. \u041D\u0430\u0441\u0442\u0440\u043E\u0439\u0442\u0435 API \u043A\u043B\u044E\u0447 \u0434\u043B\u044F \u043D\u0430\u0447\u0430\u043B\u0430 \u0440\u0430\u0431\u043E\u0442\u044B.</span>
    `;
    const setupBtn = document.createElement("button");
    setupBtn.textContent = "\u041D\u0430\u0441\u0442\u0440\u043E\u0438\u0442\u044C";
    setupBtn.type = "button";
    setupBtn.style.cssText = `
        padding: 8px 16px;
        background: ${C.COLORS.STATUS_WARNING};
        color: #000;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        font-size: 13px;
        font-weight: 600;
        white-space: nowrap;
        transition: all 0.2s ease;
    `;
    setupBtn.addEventListener("mouseenter", () => {
      setupBtn.style.opacity = "0.9";
      setupBtn.style.transform = "translateY(-1px)";
    });
    setupBtn.addEventListener("mouseleave", () => {
      setupBtn.style.opacity = "1";
      setupBtn.style.transform = "translateY(0)";
    });
    setupBtn.addEventListener("click", () => {
      chrome.runtime.sendMessage({ type: "OPEN_OPTIONS" });
    });
    container.appendChild(messageDiv);
    container.appendChild(setupBtn);
    const inputDiv = formDiv.parentElement;
    if (inputDiv && inputDiv.classList.contains("inputDiv")) {
      inputDiv.parentNode.insertBefore(container, inputDiv.nextSibling);
    } else {
      formDiv.parentNode.insertBefore(container, formDiv.nextSibling);
    }
  }
  function createTranslateInterface(formDiv, type = "main") {
    const containerId = `chat-translate-container-${type}`;
    const inputId = `chat-translate-input-${type}`;
    const buttonId = `chat-translate-btn-${type}`;
    const statusId = `chat-translate-status-${type}`;
    if (document.getElementById(containerId)) return;
    const container = document.createElement("div");
    container.id = containerId;
    container.style.cssText = `
        margin-top: 10px;
        padding: 10px;
        background: ${C.COLORS.BG_PRIMARY};
        backdrop-filter: blur(20px);
        -webkit-backdrop-filter: blur(20px);
        border-radius: 0px;
        box-shadow: 0 8px 32px 0 rgba(0, 0, 0, 0.4);
    `;
    const inputWrapper = document.createElement("div");
    inputWrapper.style.cssText = "display: flex; margin-bottom: 5px;";
    const input = document.createElement("input");
    input.type = "text";
    input.id = inputId;
    input.placeholder = `${C.TEXT.PLACEHOLDER_INPUT} (${type.toUpperCase()} \u0447\u0430\u0442)`;
    input.style.cssText = `
        flex: 1;
        padding: 12px 16px;
        background: ${C.COLORS.BG_INPUT};
        border: 1px solid ${C.COLORS.BORDER_DEFAULT};
        border-radius: 6px;
        font-size: 14px;
        margin-right: 12px;
        color: ${C.COLORS.TEXT_PRIMARY};
        outline: none;
        transition: all 0.3s ease;
    `;
    const styleId = `chat-translate-style-${type}`;
    if (!document.getElementById(styleId)) {
      const style = document.createElement("style");
      style.id = styleId;
      style.textContent = `
            #${inputId}::placeholder {
                color: ${C.COLORS.TEXT_PLACEHOLDER};
                opacity: 1;
            }
        `;
      document.head.appendChild(style);
    }
    U.addFocusEffects(
      input,
      { background: C.COLORS.BG_INPUT_FOCUS, borderColor: C.COLORS.BORDER_FOCUS },
      { background: C.COLORS.BG_INPUT, borderColor: C.COLORS.BORDER_DEFAULT }
    );
    const button = document.createElement("button");
    button.textContent = C.TEXT.TRANSLATE_BTN;
    button.id = buttonId;
    button.type = "button";
    button.style.cssText = `
        padding: 12px 24px;
        background: ${C.COLORS.BG_BUTTON};
        color: ${C.COLORS.TEXT_PRIMARY};
        border: 1px solid ${C.COLORS.BORDER_BUTTON};
        border-radius: 6px;
        cursor: pointer;
        font-size: 14px;
        white-space: nowrap;
        transition: all 0.3s ease;
        font-weight: 500;
        letter-spacing: 0.5px;
    `;
    U.addHoverEffects(
      button,
      {
        background: C.COLORS.BG_BUTTON_HOVER,
        borderColor: C.COLORS.BORDER_BUTTON_HOVER,
        transform: "translateY(-1px)",
        boxShadow: "0 4px 12px rgba(255, 255, 255, 0.1)"
      },
      {
        background: C.COLORS.BG_BUTTON,
        borderColor: C.COLORS.BORDER_BUTTON,
        transform: "translateY(0)",
        boxShadow: "none"
      }
    );
    const statusDiv = document.createElement("div");
    statusDiv.id = statusId;
    statusDiv.style.cssText = `
        font-size: 12px;
        margin-top: 8px;
        min-height: 20px;
        color: ${C.COLORS.TEXT_SECONDARY};
        font-weight: 400;
        letter-spacing: 0.3px;
    `;
    input.addEventListener("keypress", (event) => {
      if (event.key === "Enter") button.click();
    });
    button.addEventListener("click", async () => {
      const textToTranslate = input.value.trim();
      if (!textToTranslate) {
        U.setStatus(statusDiv, C.TEXT.ERR_ENTER_TEXT, "error");
        return;
      }
      button.disabled = true;
      button.textContent = C.TEXT.TRANSLATING;
      button.style.opacity = "0.6";
      U.setStatus(statusDiv, C.TEXT.TRANSLATING, "info");
      const { myLanguage, roomLanguage } = await U.getLanguageSettings();
      const result = await translateText(textToTranslate, myLanguage, roomLanguage);
      if (result.success) {
        const chatInput = findChatInput(formDiv);
        if (chatInput) {
          chatInput.textContent = result.translatedText;
          chatInput.dispatchEvent(new Event("input", { bubbles: true }));
          chatInput.dispatchEvent(new Event("change", { bubbles: true }));
          chatInput.focus();
          U.setStatus(statusDiv, `\u2713 \u041F\u0435\u0440\u0435\u0432\u043E\u0434: "${textToTranslate}" \u2192 "${result.translatedText}"`, "success");
          input.value = "";
        } else {
          U.logError("Content", "Could not find chat input field");
          U.setStatus(statusDiv, C.TEXT.ERR_NO_CHAT_INPUT, "error");
        }
      } else {
        U.logError("Content", "Translation failed:", result.error);
        U.setStatus(statusDiv, `Error: ${result.error}`, "error");
      }
      button.disabled = false;
      button.textContent = C.TEXT.TRANSLATE_BTN;
      button.style.opacity = "1";
    });
    inputWrapper.appendChild(input);
    inputWrapper.appendChild(button);
    container.appendChild(inputWrapper);
    container.appendChild(statusDiv);
    const inputDiv = formDiv.parentElement;
    if (inputDiv && inputDiv.classList.contains("inputDiv")) {
      inputDiv.parentNode.insertBefore(container, inputDiv.nextSibling);
    } else {
      formDiv.parentNode.insertBefore(container, formDiv.nextSibling);
    }
  }
  function createUI(formDiv, type = "main") {
    if (hasApiKey) {
      createTranslateInterface(formDiv, type);
    } else {
      createSetupNotice(formDiv, type);
    }
  }
  var initAttempts = 0;
  async function init() {
    initAttempts++;
    if (initAttempts === 1) {
      await checkApiKeyStatus();
      await loadTipMenu();
      await loadSessionEarnings();
      createEarningsPanel();
    }
    const forms = findChatInputForms();
    if (forms.length === 0) {
      if (initAttempts < C.TIMING.MAX_INIT_ATTEMPTS) {
        setTimeout(init, C.TIMING.PAGE_LOAD_CHECK);
      }
      return;
    }
    forms.forEach(({ form, type }) => {
      try {
        createUI(form, type);
      } catch (error) {
        U.logError("Content", `Error creating interface for ${type} chat:`, error);
      }
    });
    setupFormObserver();
    if (hasApiKey) {
      setupMessageObserver();
    }
  }
  function checkAndAddInterface(form, type) {
    const containerId = `chat-translate-container-${type}`;
    if (!document.getElementById(containerId)) {
      try {
        createUI(form, type);
      } catch (error) {
        U.logError("Content", `Error creating interface for ${type} chat:`, error);
      }
    }
  }
  function setupFormObserver() {
    U.createNodeObserver((node) => {
      const mainChatInput = node.querySelector?.(C.SELECTORS.MAIN_CHAT_INPUT);
      if (mainChatInput) {
        const form = mainChatInput.closest("form");
        if (form) checkAndAddInterface(form, "main");
      }
      const pmChatInputs = node.querySelectorAll?.(C.SELECTORS.PM_CHAT_INPUT) || [];
      pmChatInputs.forEach((pmChatInput, index) => {
        const pmForm = pmChatInput.closest("form");
        if (pmForm) {
          const pmId = pmForm.id || `pm-${Date.now()}-${index}`;
          checkAndAddInterface(pmForm, pmId);
        }
      });
      if (node.classList?.contains("theatermodeInputFieldChat")) {
        const form = node.closest("form");
        if (form) checkAndAddInterface(form, "main");
      }
      if (node.classList?.contains("theatermodeInputFieldPm")) {
        const pmForm = node.closest("form");
        if (pmForm) {
          const pmId = pmForm.id || `pm-${Date.now()}`;
          checkAndAddInterface(pmForm, pmId);
        }
      }
    });
  }
  async function processChatMessage(messageDiv) {
    if (!hasApiKey) return;
    const chatTextSpan = messageDiv.querySelector(C.SELECTORS.CHAT_MESSAGE_TEXT);
    if (!chatTextSpan) return;
    if (!chatTextSpan.classList.contains("split-mode")) return;
    const textSpan = chatTextSpan.querySelector("span");
    if (!textSpan || !textSpan.textContent.trim()) return;
    const messageContainer = chatTextSpan.parentElement;
    if (messageContainer.querySelector('[data-translation-text="true"]')) return;
    const originalText = textSpan.textContent.trim();
    const { myLanguage, roomLanguage } = await U.getLanguageSettings();
    let borderColor = C.COLORS.TRANSLATION_BORDER;
    const usernameDiv = messageDiv.querySelector('[data-testid="chat-message-username"]');
    if (usernameDiv) {
      if (usernameDiv.classList.contains("tippedTonsRecently")) {
        borderColor = "#804baa";
      } else if (usernameDiv.classList.contains("tippedALotRecently")) {
        borderColor = "#be6aff";
      } else if (usernameDiv.classList.contains("tippedRecently")) {
        borderColor = "#1e5cfb";
      } else if (usernameDiv.classList.contains("inFanclub")) {
        borderColor = "#009900";
      } else if (usernameDiv.classList.contains("hasTokens")) {
        borderColor = "#69a";
      } else if (usernameDiv.classList.contains("mod")) {
        borderColor = "#dc0000";
      }
    }
    try {
      const result = await translateText(originalText, roomLanguage, myLanguage);
      if (result.success && result.translatedText) {
        const translationElement = document.createElement("div");
        translationElement.textContent = "\u21B3 " + result.translatedText;
        translationElement.style.cssText = `
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Inter', system-ui, sans-serif;
                font-size: 1.2em;
                font-weight: 400;
                line-height: 1.65;
                letter-spacing: 0.015em;
                color: ${C.COLORS.TRANSLATION_TEXT};
                background-color: ${C.COLORS.TRANSLATION_BG};
                padding: 8px 12px;
                border-left: 6px solid ${borderColor};
                border-radius: 4px;
                max-width: 72ch;
                box-shadow: inset 0 0 5px rgba(0, 0, 0, 0.5);
                transition: background-color 0.2s ease;
                -webkit-font-smoothing: antialiased;
                -moz-osx-font-smoothing: grayscale;
            `;
        translationElement.setAttribute("data-translation-text", "true");
        messageContainer.appendChild(translationElement);
        const chatScrollContainer = messageDiv.closest(C.SELECTORS.MSG_LIST_WRAPPER);
        if (chatScrollContainer) {
          chatScrollContainer.scrollTop = chatScrollContainer.scrollHeight;
        }
      } else if (result.error) {
        U.log("Content", "Translation skipped:", result.error);
      }
    } catch (error) {
      U.logError("Content", "Translation error:", error);
    }
  }
  function setupMessageObserver() {
    U.createNodeObserver((node) => {
      if (node.getAttribute("data-testid") === "chat-message") {
        processChatMessage(node);
      } else {
        const chatMessages = node.querySelectorAll(C.SELECTORS.CHAT_MESSAGE);
        chatMessages.forEach((msg) => processChatMessage(msg));
      }
    });
  }
  var tipMenuMap = /* @__PURE__ */ new Map();
  var tipModalFontSize = 15;
  var attentionRate = 3;
  var streakMultiplier = 3.5;
  var activeModals = [];
  var MODAL_MIN_DURATION = 5e3;
  function calculateModalDuration(tokens) {
    const earnings = tokens * 0.05;
    const minutes = earnings / attentionRate;
    const ms = minutes * 60 * 1e3;
    return Math.max(MODAL_MIN_DURATION, ms);
  }
  var sessionEarnings = 0;
  var earningsPanel = null;
  var earningsDisplay = null;
  var displayedEarnings = 0;
  var modalsPaused = false;
  var pauseButton = null;
  var pageTokenCount = 0;
  var tokenCountDisplay = null;
  var tokenCountObserver = null;
  var uiScale = 0.8;
  var UI_SCALE_MIN = 0.5;
  var UI_SCALE_MAX = 2;
  var UI_SCALE_STEP = 0.1;
  var streakActive = false;
  var streakStartTime = null;
  var streakTipCount = 0;
  var streakEarnings = 0;
  var streakTimeout = null;
  var streakCurrentDuration = 0;
  var lastTipTime = null;
  var streakDisplay = null;
  var EARNINGS_COLOR_STOPS = [
    { amount: 0, color: [255, 255, 255] },
    // White - starting out
    { amount: 75, color: [96, 165, 250] },
    // Blue - warming up
    { amount: 150, color: [244, 114, 182] },
    // Pink - typical session
    { amount: 225, color: [168, 85, 247] },
    // Purple - great session
    { amount: 300, color: [239, 68, 68] }
    // Red - incredible!
  ];
  function getEarningsColor(amount) {
    let lower = EARNINGS_COLOR_STOPS[0];
    let upper = EARNINGS_COLOR_STOPS[EARNINGS_COLOR_STOPS.length - 1];
    for (let i = 0; i < EARNINGS_COLOR_STOPS.length - 1; i++) {
      if (amount >= EARNINGS_COLOR_STOPS[i].amount && amount < EARNINGS_COLOR_STOPS[i + 1].amount) {
        lower = EARNINGS_COLOR_STOPS[i];
        upper = EARNINGS_COLOR_STOPS[i + 1];
        break;
      }
    }
    if (amount >= EARNINGS_COLOR_STOPS[EARNINGS_COLOR_STOPS.length - 1].amount) {
      const gold = EARNINGS_COLOR_STOPS[EARNINGS_COLOR_STOPS.length - 1].color;
      return { hex: `rgb(${gold[0]}, ${gold[1]}, ${gold[2]})`, rgb: gold.join(", ") };
    }
    const range = upper.amount - lower.amount;
    const progress = (amount - lower.amount) / range;
    const r = Math.round(lower.color[0] + (upper.color[0] - lower.color[0]) * progress);
    const g = Math.round(lower.color[1] + (upper.color[1] - lower.color[1]) * progress);
    const b = Math.round(lower.color[2] + (upper.color[2] - lower.color[2]) * progress);
    return { hex: `rgb(${r}, ${g}, ${b})`, rgb: `${r}, ${g}, ${b}` };
  }
  function getStreakColor(earnings) {
    const progress = Math.min(1, earnings / 10);
    const hue = 180 + progress * 140;
    return `hsl(${hue}, 100%, 65%)`;
  }
  function getHeartbeatScale(earnings) {
    const minScale = 1.25;
    const maxScale = 1.5;
    const scale = Math.min(maxScale, minScale + earnings * 0.0125);
    return scale;
  }
  function getRoomId() {
    const path = window.location.pathname;
    const match = path.match(/^\/(?:b\/)?([^/]+)/);
    return match ? match[1] : "unknown";
  }
  function getStorageKey(key) {
    const roomId = getRoomId();
    return `${roomId}_${key}`;
  }
  async function loadSessionEarnings() {
    try {
      const key = getStorageKey("sessionEarnings");
      const result = await chrome.storage.session.get([key, "uiScale"]);
      if (result[key]) {
        sessionEarnings = result[key];
        displayedEarnings = sessionEarnings;
      }
      if (result.uiScale) {
        uiScale = result.uiScale;
      }
      U.log("Content", `Loaded earnings for room ${getRoomId()}: $${sessionEarnings.toFixed(2)}, UI scale: ${uiScale}`);
    } catch (error) {
      U.logError("Content", "Failed to load session earnings:", error);
    }
  }
  async function saveSessionEarnings() {
    try {
      const key = getStorageKey("sessionEarnings");
      await chrome.storage.session.set({ [key]: sessionEarnings });
    } catch (error) {
      U.logError("Content", "Failed to save session earnings:", error);
    }
  }
  function resetSessionEarnings() {
    sessionEarnings = 0;
    displayedEarnings = 0;
    saveSessionEarnings();
    if (earningsDisplay) {
      earningsDisplay.textContent = "$0.00";
      const resetColor = getEarningsColor(0);
      earningsDisplay.style.color = resetColor.hex;
    }
    if (streakActive) {
      streakActive = false;
      if (streakTimeout) {
        clearTimeout(streakTimeout);
        streakTimeout = null;
      }
      if (heartbeatInterval) {
        clearInterval(heartbeatInterval);
        heartbeatInterval = null;
      }
      if (streakDisplay) {
        clearInterval(streakDisplay._interval);
        streakDisplay.style.display = "none";
        streakDisplay._initialized = false;
      }
    }
    U.log("Content", `Reset earnings and streak for room ${getRoomId()}`);
  }
  async function createEarningsPanel() {
    if (earningsPanel) return;
    let savedPosition = null;
    try {
      const positionKey = getStorageKey("earningsPanelPosition");
      const result = await chrome.storage.session.get([positionKey, "modalsPaused"]);
      savedPosition = result[positionKey];
      if (result.modalsPaused !== void 0) {
        modalsPaused = result.modalsPaused;
      }
    } catch (e) {
      U.log("Content", "Could not load panel state:", e);
    }
    earningsPanel = document.createElement("div");
    const positionStyle = savedPosition ? `top: ${savedPosition.top}px; left: ${savedPosition.left}px;` : `top: 20px; right: 20px;`;
    earningsPanel.style.cssText = `
        position: fixed;
        ${positionStyle}
        z-index: 10001;
        background: rgba(17, 24, 39, 0.9);
        backdrop-filter: blur(12px);
        -webkit-backdrop-filter: blur(12px);
        border: 1px solid rgba(255, 255, 255, 0.2);
        border-radius: 12px;
        padding: 20px 28px;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Inter', system-ui, sans-serif;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4);
        transition: border-color 0.3s ease, box-shadow 0.3s ease;
        cursor: grab;
        user-select: none;
    `;
    let isDragging = false;
    let dragStartX, dragStartY, panelStartLeft, panelStartTop;
    earningsPanel.addEventListener("mousedown", (e) => {
      if (e.target.tagName === "BUTTON" || e.target.closest("button")) return;
      isDragging = true;
      earningsPanel.style.cursor = "grabbing";
      earningsPanel.style.transition = "border-color 0.3s ease, box-shadow 0.3s ease";
      dragStartX = e.clientX;
      dragStartY = e.clientY;
      const computedStyle = window.getComputedStyle(earningsPanel);
      const currentLeft = parseFloat(computedStyle.left);
      const currentTop = parseFloat(computedStyle.top);
      const currentRight = parseFloat(computedStyle.right);
      if (isNaN(currentLeft) || computedStyle.left === "auto") {
        const rect = earningsPanel.getBoundingClientRect();
        const actualWidth = rect.width / uiScale;
        panelStartLeft = window.innerWidth - currentRight - actualWidth;
      } else {
        panelStartLeft = currentLeft;
      }
      panelStartTop = isNaN(currentTop) ? 20 : currentTop;
      earningsPanel.style.right = "auto";
      earningsPanel.style.left = panelStartLeft + "px";
      earningsPanel.style.top = panelStartTop + "px";
      e.preventDefault();
    });
    document.addEventListener("mousemove", (e) => {
      if (!isDragging) return;
      const deltaX = e.clientX - dragStartX;
      const deltaY = e.clientY - dragStartY;
      let newLeft = panelStartLeft + deltaX;
      let newTop = panelStartTop + deltaY;
      const rect = earningsPanel.getBoundingClientRect();
      const scaledWidth = rect.width;
      const scaledHeight = rect.height;
      newLeft = Math.max(0, Math.min(newLeft, window.innerWidth - scaledWidth));
      newTop = Math.max(0, Math.min(newTop, window.innerHeight - scaledHeight));
      earningsPanel.style.left = newLeft + "px";
      earningsPanel.style.top = newTop + "px";
    });
    document.addEventListener("mouseup", () => {
      if (!isDragging) return;
      isDragging = false;
      earningsPanel.style.cursor = "grab";
      const computedStyle = window.getComputedStyle(earningsPanel);
      const position = {
        top: parseFloat(computedStyle.top),
        left: parseFloat(computedStyle.left)
      };
      const positionKey = getStorageKey("earningsPanelPosition");
      chrome.storage.session.set({ [positionKey]: position }).catch((e) => {
        U.log("Content", "Could not save panel position:", e);
      });
    });
    const row = document.createElement("div");
    row.style.cssText = `
        display: flex;
        align-items: center;
        justify-content: space-between;
    `;
    const initialColor = getEarningsColor(displayedEarnings);
    earningsDisplay = document.createElement("div");
    earningsDisplay.textContent = `$${displayedEarnings.toFixed(2)}`;
    earningsDisplay.style.cssText = `
        font-size: 42px;
        font-weight: 700;
        color: ${initialColor.hex};
        font-variant-numeric: tabular-nums;
        text-shadow: 0 0 20px rgba(${initialColor.rgb}, 0);
        transition: color 0.5s ease, text-shadow 0.3s ease;
    `;
    const zoomContainer = document.createElement("div");
    zoomContainer.style.cssText = `
        display: flex;
        align-items: center;
        gap: 2px;
    `;
    const bellIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 8A6 6 0 0 0 6 8c0 7-3 9-3 9h18s-3-2-3-9"/><path d="M13.73 21a2 2 0 0 1-3.46 0"/></svg>`;
    const bellOffIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M13.73 21a2 2 0 0 1-3.46 0"/><path d="M18.63 13A17.89 17.89 0 0 1 18 8"/><path d="M6.26 6.26A5.86 5.86 0 0 0 6 8c0 7-3 9-3 9h14"/><path d="M18 8a6 6 0 0 0-9.33-5"/><line x1="1" y1="1" x2="23" y2="23"/></svg>`;
    const toggleWrapper = document.createElement("div");
    toggleWrapper.style.cssText = `position: relative;`;
    pauseButton = document.createElement("button");
    pauseButton.innerHTML = modalsPaused ? bellOffIcon : bellIcon;
    pauseButton.style.cssText = `
        background: none;
        border: none;
        color: ${modalsPaused ? "rgba(239, 68, 68, 0.8)" : "rgba(255, 255, 255, 0.3)"};
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
        border-radius: 4px;
    `;
    const toggleTooltip = document.createElement("div");
    toggleTooltip.textContent = modalsPaused ? "\u0412\u043A\u043B\u044E\u0447\u0438\u0442\u044C \u0443\u0432\u0435\u0434\u043E\u043C\u043B\u0435\u043D\u0438\u044F" : "\u0412\u044B\u043A\u043B\u044E\u0447\u0438\u0442\u044C \u0443\u0432\u0435\u0434\u043E\u043C\u043B\u0435\u043D\u0438\u044F";
    toggleTooltip.style.cssText = `
        position: absolute;
        bottom: 100%;
        left: 50%;
        transform: translateX(-50%);
        background: rgba(0, 0, 0, 0.9);
        color: white;
        padding: 6px 10px;
        border-radius: 6px;
        font-size: 12px;
        white-space: nowrap;
        pointer-events: none;
        opacity: 0;
        transition: opacity 0.15s ease;
        margin-bottom: 6px;
    `;
    pauseButton._tooltip = toggleTooltip;
    pauseButton.addEventListener("mouseenter", () => {
      pauseButton.style.color = modalsPaused ? "rgba(239, 68, 68, 1)" : "rgba(255, 255, 255, 0.8)";
      pauseButton.style.background = "rgba(255, 255, 255, 0.1)";
      toggleTooltip.style.opacity = "1";
    });
    pauseButton.addEventListener("mouseleave", () => {
      pauseButton.style.color = modalsPaused ? "rgba(239, 68, 68, 0.8)" : "rgba(255, 255, 255, 0.3)";
      pauseButton.style.background = "none";
      toggleTooltip.style.opacity = "0";
    });
    pauseButton.addEventListener("click", toggleModalsPaused);
    toggleWrapper.appendChild(pauseButton);
    toggleWrapper.appendChild(toggleTooltip);
    const resetWrapper = document.createElement("div");
    resetWrapper.style.cssText = `position: relative;`;
    const resetBtn = document.createElement("button");
    resetBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 12a9 9 0 1 0 9-9 9.75 9.75 0 0 0-6.74 2.74L3 8"/><path d="M3 3v5h5"/></svg>`;
    resetBtn.style.cssText = `
        background: none;
        border: none;
        color: rgba(255, 255, 255, 0.3);
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
        border-radius: 4px;
    `;
    const resetTooltip = document.createElement("div");
    resetTooltip.textContent = "\u0421\u0431\u0440\u043E\u0441\u0438\u0442\u044C \u0437\u0430\u0440\u0430\u0431\u043E\u0442\u043E\u043A";
    resetTooltip.style.cssText = `
        position: absolute;
        bottom: 100%;
        left: 50%;
        transform: translateX(-50%);
        background: rgba(0, 0, 0, 0.9);
        color: white;
        padding: 6px 10px;
        border-radius: 6px;
        font-size: 12px;
        white-space: nowrap;
        pointer-events: none;
        opacity: 0;
        transition: opacity 0.15s ease;
        margin-bottom: 6px;
    `;
    resetBtn.addEventListener("mouseenter", () => {
      resetBtn.style.color = "rgba(255, 255, 255, 0.8)";
      resetBtn.style.background = "rgba(255, 255, 255, 0.1)";
      resetTooltip.style.opacity = "1";
    });
    resetBtn.addEventListener("mouseleave", () => {
      resetBtn.style.color = "rgba(255, 255, 255, 0.3)";
      resetBtn.style.background = "none";
      resetTooltip.style.opacity = "0";
    });
    resetBtn.addEventListener("click", resetSessionEarnings);
    resetWrapper.appendChild(resetBtn);
    resetWrapper.appendChild(resetTooltip);
    const settingsWrapper = document.createElement("div");
    settingsWrapper.style.cssText = `position: relative;`;
    const settingsBtn = document.createElement("button");
    settingsBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg>`;
    settingsBtn.style.cssText = `
        background: none;
        border: none;
        color: rgba(255, 255, 255, 0.3);
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
        border-radius: 4px;
    `;
    const settingsTooltip = document.createElement("div");
    settingsTooltip.textContent = "\u041D\u0430\u0441\u0442\u0440\u043E\u0439\u043A\u0438";
    settingsTooltip.style.cssText = `
        position: absolute;
        bottom: 100%;
        left: 50%;
        transform: translateX(-50%);
        background: rgba(0, 0, 0, 0.9);
        color: white;
        padding: 6px 10px;
        border-radius: 6px;
        font-size: 12px;
        white-space: nowrap;
        pointer-events: none;
        opacity: 0;
        transition: opacity 0.15s ease;
        margin-bottom: 6px;
    `;
    settingsBtn.addEventListener("mouseenter", () => {
      settingsBtn.style.color = "rgba(255, 255, 255, 0.8)";
      settingsBtn.style.background = "rgba(255, 255, 255, 0.1)";
      settingsTooltip.style.opacity = "1";
    });
    settingsBtn.addEventListener("mouseleave", () => {
      settingsBtn.style.color = "rgba(255, 255, 255, 0.3)";
      settingsBtn.style.background = "none";
      settingsTooltip.style.opacity = "0";
    });
    settingsBtn.addEventListener("click", () => {
      chrome.runtime.sendMessage({ type: "OPEN_OPTIONS" });
    });
    settingsWrapper.appendChild(settingsBtn);
    settingsWrapper.appendChild(settingsTooltip);
    const scaleWrapper = document.createElement("div");
    scaleWrapper.style.cssText = `
        display: flex;
        align-items: center;
        gap: 2px;
        margin-left: 16px;
        padding-left: 16px;
        border-left: 1px solid rgba(255, 255, 255, 0.15);
    `;
    const zoomOutIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="8"/><line x1="21" y1="21" x2="16.65" y2="16.65"/><line x1="8" y1="11" x2="14" y2="11"/></svg>`;
    const zoomInIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="8"/><line x1="21" y1="21" x2="16.65" y2="16.65"/><line x1="11" y1="8" x2="11" y2="14"/><line x1="8" y1="11" x2="14" y2="11"/></svg>`;
    const minusBtn = document.createElement("button");
    minusBtn.innerHTML = zoomOutIcon;
    minusBtn.style.cssText = `
        background: none;
        border: none;
        color: rgba(255, 255, 255, 0.3);
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
        border-radius: 4px;
    `;
    minusBtn.addEventListener("mouseenter", () => {
      minusBtn.style.color = "rgba(255, 255, 255, 0.8)";
      minusBtn.style.background = "rgba(255, 255, 255, 0.1)";
    });
    minusBtn.addEventListener("mouseleave", () => {
      minusBtn.style.color = "rgba(255, 255, 255, 0.3)";
      minusBtn.style.background = "none";
    });
    minusBtn.addEventListener("click", () => adjustUiScale(-UI_SCALE_STEP));
    const plusBtn = document.createElement("button");
    plusBtn.innerHTML = zoomInIcon;
    plusBtn.style.cssText = `
        background: none;
        border: none;
        color: rgba(255, 255, 255, 0.3);
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
        border-radius: 4px;
    `;
    plusBtn.addEventListener("mouseenter", () => {
      plusBtn.style.color = "rgba(255, 255, 255, 0.8)";
      plusBtn.style.background = "rgba(255, 255, 255, 0.1)";
    });
    plusBtn.addEventListener("mouseleave", () => {
      plusBtn.style.color = "rgba(255, 255, 255, 0.3)";
      plusBtn.style.background = "none";
    });
    plusBtn.addEventListener("click", () => adjustUiScale(UI_SCALE_STEP));
    scaleWrapper.appendChild(minusBtn);
    scaleWrapper.appendChild(plusBtn);
    zoomContainer.appendChild(scaleWrapper);
    row.appendChild(earningsDisplay);
    row.appendChild(zoomContainer);
    earningsPanel.appendChild(row);
    const bottomRow = document.createElement("div");
    bottomRow.style.cssText = `
        margin-top: 12px;
        padding-top: 12px;
        border-top: 1px solid rgba(255, 255, 255, 0.1);
        display: flex;
        align-items: center;
        justify-content: space-between;
    `;
    const controlsContainer = document.createElement("div");
    controlsContainer.style.cssText = `
        display: flex;
        align-items: center;
        gap: 4px;
    `;
    controlsContainer.appendChild(toggleWrapper);
    controlsContainer.appendChild(resetWrapper);
    controlsContainer.appendChild(settingsWrapper);
    tokenCountDisplay = document.createElement("div");
    tokenCountDisplay.style.cssText = `
        opacity: 0;
        transition: opacity 0.3s ease;
        position: relative;
    `;
    const tokenDollar = document.createElement("span");
    tokenDollar.className = "token-dollar";
    tokenDollar.style.cssText = `
        font-size: 14px;
        font-weight: 600;
        color: rgba(255, 255, 255, 0.6);
        font-variant-numeric: tabular-nums;
        transition: color 0.3s ease, text-shadow 0.3s ease;
    `;
    tokenDollar.textContent = "~$0.00";
    tokenCountDisplay.appendChild(tokenDollar);
    bottomRow.appendChild(controlsContainer);
    bottomRow.appendChild(tokenCountDisplay);
    earningsPanel.appendChild(bottomRow);
    document.body.appendChild(earningsPanel);
    applyUiScale();
    observePageTokenCount();
  }
  function applyUiScale() {
    if (earningsPanel) {
      earningsPanel.style.transform = `scale(${uiScale})`;
      earningsPanel.style.transformOrigin = "top right";
    }
  }
  var previewModalTimeout = null;
  var previewModalInfo = null;
  function adjustUiScale(delta) {
    const newScale = Math.round((uiScale + delta) * 10) / 10;
    uiScale = Math.max(UI_SCALE_MIN, Math.min(UI_SCALE_MAX, newScale));
    applyUiScale();
    chrome.storage.session.set({ uiScale }).catch((e) => {
      U.log("Content", "Could not save UI scale:", e);
    });
    U.log("Content", `UI scale: ${uiScale}`);
    showScalePreviewModal();
  }
  function showScalePreviewModal() {
    if (previewModalInfo) {
      removeModal(previewModalInfo);
      previewModalInfo = null;
    }
    if (previewModalTimeout) {
      clearTimeout(previewModalTimeout);
    }
    const savedPaused = modalsPaused;
    modalsPaused = false;
    showTipModal(
      100,
      "preview_user",
      "\u041F\u0440\u0438\u043C\u0435\u0440 \u0443\u0432\u0435\u0434\u043E\u043C\u043B\u0435\u043D\u0438\u044F \u043E \u0447\u0430\u0435\u0432\u044B\u0445",
      3e3,
      { hasTokens: true }
    );
    modalsPaused = savedPaused;
    previewModalInfo = activeModals[activeModals.length - 1];
    previewModalTimeout = setTimeout(() => {
      previewModalInfo = null;
      previewModalTimeout = null;
    }, 3e3);
  }
  function toggleModalsPaused() {
    modalsPaused = !modalsPaused;
    const bellIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 8A6 6 0 0 0 6 8c0 7-3 9-3 9h18s-3-2-3-9"/><path d="M13.73 21a2 2 0 0 1-3.46 0"/></svg>`;
    const bellOffIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M13.73 21a2 2 0 0 1-3.46 0"/><path d="M18.63 13A17.89 17.89 0 0 1 18 8"/><path d="M6.26 6.26A5.86 5.86 0 0 0 6 8c0 7-3 9-3 9h14"/><path d="M18 8a6 6 0 0 0-9.33-5"/><line x1="1" y1="1" x2="23" y2="23"/></svg>`;
    if (pauseButton) {
      pauseButton.innerHTML = modalsPaused ? bellOffIcon : bellIcon;
      pauseButton.style.color = modalsPaused ? "rgba(239, 68, 68, 0.8)" : "rgba(255, 255, 255, 0.3)";
      if (pauseButton._tooltip) {
        pauseButton._tooltip.textContent = modalsPaused ? "\u0412\u043A\u043B\u044E\u0447\u0438\u0442\u044C \u0443\u0432\u0435\u0434\u043E\u043C\u043B\u0435\u043D\u0438\u044F" : "\u0412\u044B\u043A\u043B\u044E\u0447\u0438\u0442\u044C \u0443\u0432\u0435\u0434\u043E\u043C\u043B\u0435\u043D\u0438\u044F";
      }
    }
    chrome.storage.session.set({ modalsPaused }).catch((e) => {
      U.log("Content", "Could not save pause state:", e);
    });
    U.log("Content", `Modals ${modalsPaused ? "paused" : "resumed"}`);
  }
  function observePageTokenCount() {
    const tokenSpan = document.querySelector(".tokencount");
    if (!tokenSpan) {
      setTimeout(observePageTokenCount, 2e3);
      return;
    }
    const initialValue = parseInt(tokenSpan.textContent.replace(/,/g, ""), 10) || 0;
    updateTokenCountDisplay(initialValue, false);
    if (tokenCountObserver) {
      tokenCountObserver.disconnect();
    }
    tokenCountObserver = new MutationObserver((mutations) => {
      for (const mutation of mutations) {
        if (mutation.type === "characterData" || mutation.type === "childList") {
          const newValue = parseInt(tokenSpan.textContent.replace(/,/g, ""), 10) || 0;
          if (newValue !== pageTokenCount) {
            const increased = newValue > pageTokenCount;
            updateTokenCountDisplay(newValue, increased);
          }
        }
      }
    });
    tokenCountObserver.observe(tokenSpan, {
      characterData: true,
      childList: true,
      subtree: true
    });
    U.log("Content", `Token count observer started, initial: ${initialValue}`);
  }
  function updateTokenCountDisplay(newValue, animate = true) {
    const oldValue = pageTokenCount;
    pageTokenCount = newValue;
    if (!tokenCountDisplay) return;
    const tokenDollar = tokenCountDisplay.querySelector(".token-dollar");
    if (!tokenDollar) return;
    tokenCountDisplay.style.opacity = "1";
    const dollarValue = (newValue * 0.05).toFixed(2);
    if (animate && newValue > oldValue) {
      animateTokenIncrease(oldValue, newValue, tokenDollar);
    } else {
      tokenDollar.textContent = `~$${dollarValue}`;
    }
  }
  function animateTokenIncrease(fromValue, toValue, dollarElement) {
    const duration = 1500;
    const startTime = performance.now();
    const difference = toValue - fromValue;
    if (tokenCountDisplay) {
      const floater = document.createElement("div");
      const addedTokens = toValue - fromValue;
      const addedDollars = (addedTokens * 0.05).toFixed(2);
      floater.textContent = `+$${addedDollars}`;
      floater.style.cssText = `
            position: absolute;
            right: 0;
            bottom: 4px;
            color: rgba(255, 255, 255, 0.7);
            font-size: 11px;
            font-weight: 600;
            opacity: 1;
            transform: translateY(0);
            transition: all 1.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            pointer-events: none;
        `;
      tokenCountDisplay.appendChild(floater);
      requestAnimationFrame(() => {
        floater.style.transform = "translateY(-25px)";
        floater.style.opacity = "0";
      });
      setTimeout(() => floater.remove(), 1500);
    }
    dollarElement.style.color = "rgba(255, 255, 255, 0.9)";
    dollarElement.style.textShadow = "0 0 8px rgba(255, 255, 255, 0.3)";
    function update(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      const eased = 1 - Math.pow(1 - progress, 3);
      const currentValue = Math.round(fromValue + difference * eased);
      const currentDollar = (currentValue * 0.05).toFixed(2);
      dollarElement.textContent = `~$${currentDollar}`;
      if (progress < 1) {
        requestAnimationFrame(update);
      } else {
        dollarElement.textContent = `~$${(toValue * 0.05).toFixed(2)}`;
        setTimeout(() => {
          dollarElement.style.color = "rgba(255, 255, 255, 0.6)";
          dollarElement.style.textShadow = "none";
        }, 300);
      }
    }
    requestAnimationFrame(update);
  }
  function animateEarningsUpdate(addedAmount) {
    if (!earningsPanel || !earningsDisplay) {
      createEarningsPanel();
    }
    const targetEarnings = sessionEarnings;
    const startEarnings = displayedEarnings;
    const duration = 3200;
    const startTime = performance.now();
    const floaterDuration = 2500;
    const targetColor = getEarningsColor(targetEarnings);
    earningsPanel.style.borderColor = `rgba(${targetColor.rgb}, 0.6)`;
    earningsPanel.style.boxShadow = `0 8px 32px rgba(0, 0, 0, 0.4), 0 0 20px rgba(${targetColor.rgb}, 0.3)`;
    earningsDisplay.style.textShadow = `0 0 20px rgba(${targetColor.rgb}, 0.5)`;
    const floater = document.createElement("div");
    floater.textContent = `+$${addedAmount.toFixed(2)}`;
    floater.style.cssText = `
        position: absolute;
        right: 36px;
        top: 50%;
        color: ${targetColor.hex};
        font-size: 20px;
        font-weight: 600;
        opacity: 1;
        transform: translateY(-50%);
        transition: all ${floaterDuration}ms cubic-bezier(0.25, 0.46, 0.45, 0.94);
        pointer-events: none;
    `;
    earningsPanel.appendChild(floater);
    requestAnimationFrame(() => {
      floater.style.transform = "translateY(-100px)";
      floater.style.opacity = "0";
    });
    setTimeout(() => floater.remove(), floaterDuration);
    function updateCounter(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      const eased = 1 - Math.pow(1 - progress, 3);
      displayedEarnings = startEarnings + (targetEarnings - startEarnings) * eased;
      earningsDisplay.textContent = `$${displayedEarnings.toFixed(2)}`;
      const currentColor = getEarningsColor(displayedEarnings);
      earningsDisplay.style.color = currentColor.hex;
      if (progress < 1) {
        requestAnimationFrame(updateCounter);
      } else {
        displayedEarnings = targetEarnings;
        earningsDisplay.textContent = `$${targetEarnings.toFixed(2)}`;
        const finalColor = getEarningsColor(targetEarnings);
        earningsDisplay.style.color = finalColor.hex;
        setTimeout(() => {
          earningsPanel.style.borderColor = "rgba(255, 255, 255, 0.2)";
          earningsPanel.style.boxShadow = "0 8px 32px rgba(0, 0, 0, 0.4)";
          earningsDisplay.style.textShadow = `0 0 20px rgba(${finalColor.rgb}, 0)`;
        }, 300);
      }
    }
    requestAnimationFrame(updateCounter);
  }
  function addEarnings(tokens) {
    const earned = tokens * 0.05;
    sessionEarnings += earned;
    saveSessionEarnings();
    animateEarningsUpdate(earned);
  }
  var streakLastTipTime = null;
  var streakBounceIntensity = 1;
  function createStreakDisplay() {
    if (streakDisplay || !earningsPanel) return;
    if (!document.getElementById("streak-animations")) {
      const style = document.createElement("style");
      style.id = "streak-animations";
      style.textContent = `
            @keyframes streak-pulse {
                0%, 100% { opacity: 0.15; }
                50% { opacity: 0.25; }
            }
            @keyframes streak-bounce {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(var(--bounce-scale, 1.05)); }
            }
            @keyframes heart-float {
                0% { transform: scale(1); }
                50% { transform: scale(1); }
                85% { transform: scale(var(--heart-scale, 1.4)); }
                100% { transform: scale(var(--heart-scale, 1.5)); }
            }
            .streak-heart {
                position: relative;
            }
            .streak-heart-float {
                position: absolute;
                top: 0;
                left: 0;
                animation: heart-float 0.5s ease-out forwards;
                pointer-events: none;
            }
        `;
      document.head.appendChild(style);
    }
    streakDisplay = document.createElement("div");
    streakDisplay.style.cssText = `
        margin-top: 10px;
        padding: 14px 12px 12px 12px;
        border-radius: 8px;
        display: none;
        position: relative;
    `;
    earningsPanel.appendChild(streakDisplay);
  }
  function updateStreakDisplay() {
    if (!streakDisplay || !streakActive) return;
    const elapsed = Date.now() - streakLastTipTime;
    const progress = Math.max(0, 1 - elapsed / streakCurrentDuration);
    const color = getEarningsColor(sessionEarnings);
    streakDisplay.style.background = `rgba(${color.rgb}, 0.15)`;
    if (!streakDisplay._initialized) {
      streakDisplay.innerHTML = `
            <div class="streak-progress" style="
                position: absolute;
                top: 0;
                left: 0;
                height: 3px;
                transition: width 1s linear;
            "></div>
            <div class="streak-bg" style="
                position: absolute;
                inset: 0;
                animation: streak-pulse 2s ease-in-out infinite;
            "></div>
            <div style="position: relative; display: flex; align-items: center; justify-content: center; gap: 8px; overflow: visible;">
                <div class="streak-heart-container" style="position: relative; width: 20px; height: 20px; overflow: visible; margin-right: 8px;">
                    <svg class="streak-heart" viewBox="0 0 24 24" width="20" height="20" style="position: absolute; top: 0; left: 0;">
                        <path d="M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 5.42 4.42 3 7.5 3c1.74 0 3.41.81 4.5 2.09C13.09 3.81 14.76 3 16.5 3 19.58 3 22 5.42 22 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z"/>
                    </svg>
                </div>
                <div class="streak-text" style="font-size: 18px; font-weight: 700; color: white;"></div>
            </div>
        `;
      streakDisplay._initialized = true;
    }
    const progressBar = streakDisplay.querySelector(".streak-progress");
    const bgDiv = streakDisplay.querySelector(".streak-bg");
    const heart = streakDisplay.querySelector(".streak-heart");
    const text = streakDisplay.querySelector(".streak-text");
    if (progressBar) {
      progressBar.style.width = `${progress * 100}%`;
      progressBar.style.background = color.hex;
    }
    if (bgDiv) {
      bgDiv.style.background = `rgba(${color.rgb}, 0.15)`;
    }
    if (text) {
      text.textContent = `${streakTipCount} tips \u2022 $${streakEarnings.toFixed(2)}`;
    }
  }
  var heartbeatInterval = null;
  var HEARTBEAT_SPEED = 0.5;
  function spawnFloatingHeart() {
    if (!streakDisplay) return;
    const container = streakDisplay.querySelector(".streak-heart-container");
    if (!container) return;
    const color = getStreakColor(streakEarnings);
    const scale = getHeartbeatScale(streakEarnings);
    const svgNS = "http://www.w3.org/2000/svg";
    const floater = document.createElementNS(svgNS, "svg");
    floater.setAttribute("class", "streak-heart-float");
    floater.setAttribute("viewBox", "0 0 24 24");
    floater.setAttribute("width", "20");
    floater.setAttribute("height", "20");
    floater.style.fill = color;
    floater.style.setProperty("--heart-scale", scale);
    const path = document.createElementNS(svgNS, "path");
    path.setAttribute("d", "M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 5.42 4.42 3 7.5 3c1.74 0 3.41.81 4.5 2.09C13.09 3.81 14.76 3 16.5 3 19.58 3 22 5.42 22 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z");
    floater.appendChild(path);
    container.appendChild(floater);
    setTimeout(() => floater.remove(), HEARTBEAT_SPEED * 1e3);
  }
  function updateStreakHeart() {
    if (!streakDisplay) return;
    const heart = streakDisplay.querySelector(".streak-heart");
    if (heart) {
      heart.style.fill = getStreakColor(streakEarnings);
    }
    if (!heartbeatInterval) {
      heartbeatInterval = setInterval(spawnFloatingHeart, HEARTBEAT_SPEED * 1e3);
      spawnFloatingHeart();
    }
  }
  function bounceStreak() {
    if (!streakDisplay) return;
    streakBounceIntensity = Math.min(1.15, 1 + streakTipCount * 0.02);
    streakDisplay.style.setProperty("--bounce-scale", streakBounceIntensity);
    streakDisplay.style.animation = "none";
    streakDisplay.offsetHeight;
    streakDisplay.style.animation = `streak-bounce 0.5s ease-out`;
  }
  function startStreak(earned) {
    streakActive = true;
    streakStartTime = Date.now();
    streakLastTipTime = Date.now();
    streakTipCount = 1;
    streakEarnings = earned;
    streakBounceIntensity = 1;
    createStreakDisplay();
    streakDisplay.style.display = "block";
    updateStreakDisplay();
    bounceStreak();
    streakDisplay._interval = setInterval(updateStreakDisplay, 1e3);
  }
  function formatDuration(ms) {
    const seconds = Math.floor(ms / 1e3);
    const minutes = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return minutes > 0 ? `${minutes}m ${secs}s` : `${secs}s`;
  }
  function endStreak() {
    if (!streakActive) return;
    const duration = Date.now() - streakStartTime;
    const finalEarnings = streakEarnings;
    const finalTipCount = streakTipCount;
    const color = getEarningsColor(sessionEarnings);
    if (streakDisplay) {
      clearInterval(streakDisplay._interval);
      if (heartbeatInterval) {
        clearInterval(heartbeatInterval);
        heartbeatInterval = null;
      }
      streakDisplay.style.background = `rgba(${color.rgb}, 0.2)`;
      streakDisplay.style.border = `2px solid ${color.hex}`;
      streakDisplay.style.animation = "streak-end-pulse 1.5s ease-in-out infinite";
      if (!document.getElementById("streak-end-animation")) {
        const style = document.createElement("style");
        style.id = "streak-end-animation";
        style.textContent = `
                @keyframes streak-end-pulse {
                    0%, 100% { box-shadow: 0 0 8px rgba(${color.rgb}, 0.4); }
                    50% { box-shadow: 0 0 20px rgba(${color.rgb}, 0.7); }
                }
            `;
        document.head.appendChild(style);
      }
      streakDisplay.innerHTML = `
            <div style="display: flex; flex-direction: column; align-items: center; padding: 10px 16px; gap: 4px;">
                <span style="font-size: 13px; color: rgba(255,255,255,0.7);">\u{1F389} \u0421\u0435\u0440\u0438\u044F \u0437\u0430\u0432\u0435\u0440\u0448\u0435\u043D\u0430</span>
                <span style="font-size: 24px; font-weight: 700; color: white;">$${finalEarnings.toFixed(2)}</span>
                <span style="font-size: 12px; color: rgba(255,255,255,0.5);">${finalTipCount} \u0447\u0430\u0435\u0432\u044B\u0445 \u0437\u0430 ${formatDuration(duration)}</span>
            </div>
        `;
      streakDisplay._initialized = false;
      setTimeout(() => {
        if (streakDisplay) {
          streakDisplay.style.transition = "opacity 0.5s ease";
          streakDisplay.style.opacity = "0";
          setTimeout(() => {
            if (streakDisplay) {
              streakDisplay.style.display = "none";
              streakDisplay.style.opacity = "1";
              streakDisplay.style.transition = "";
              streakDisplay.style.border = "";
              streakDisplay.style.animation = "";
              streakDisplay.style.boxShadow = "";
            }
          }, 500);
        }
      }, 8e3);
    }
    streakActive = false;
    streakStartTime = null;
    streakLastTipTime = null;
    streakTipCount = 0;
    streakEarnings = 0;
    streakTimeout = null;
    streakBounceIntensity = 1;
  }
  function recordTip(tokens) {
    const earned = tokens * 0.05;
    const now = Date.now();
    U.log("Content", `Recording tip: ${tokens} tokens, streak active: ${streakActive}, lastTipTime: ${lastTipTime ? now - lastTipTime + "ms ago" : "none"}`);
    if (streakTimeout) {
      clearTimeout(streakTimeout);
    }
    if (streakActive) {
      streakTipCount++;
      streakEarnings += earned;
      streakLastTipTime = now;
      updateStreakDisplay();
      updateStreakHeart();
      bounceStreak();
      U.log("Content", `Streak continued: ${streakTipCount} tips, $${streakEarnings.toFixed(2)}`);
    } else {
      startStreak(earned);
      streakTipCount = 1;
      updateStreakHeart();
      U.log("Content", `Streak STARTED! 1 tip, $${streakEarnings.toFixed(2)}`);
    }
    lastTipTime = now;
    streakCurrentDuration = calculateModalDuration(tokens) * streakMultiplier;
    streakLastTipTime = now;
    streakTimeout = setTimeout(endStreak, streakCurrentDuration);
    U.log("Content", `Streak timeout set to ${streakCurrentDuration / 1e3}s based on ${tokens} tokens`);
  }
  function parseTipMenu(tipMenuText) {
    const map = /* @__PURE__ */ new Map();
    if (!tipMenuText) return map;
    const lines = tipMenuText.split("\n");
    for (const line of lines) {
      const trimmed = line.trim();
      if (!trimmed) continue;
      const match = trimmed.match(/^(\d+)(?::|[\s-]+)\s*(.+)$/);
      if (match) {
        const amount = parseInt(match[1], 10);
        let description = match[2].trim();
        let durationOverride = null;
        const durationMatch = description.match(/^(.+?)\s*\|\s*(\d+)\s*$/);
        if (durationMatch) {
          description = durationMatch[1].trim();
          durationOverride = parseInt(durationMatch[2], 10) * 1e3;
        }
        map.set(amount, { description, durationOverride });
      }
    }
    return map;
  }
  async function loadTipMenu() {
    try {
      const result = await chrome.storage.sync.get(["tipMenu", "tipModalFontSize", "attentionRate", "streakMultiplier"]);
      tipMenuMap = parseTipMenu(result.tipMenu || "");
      tipModalFontSize = parseInt(result.tipModalFontSize, 10) || 15;
      attentionRate = parseFloat(result.attentionRate) || 3;
      streakMultiplier = parseFloat(result.streakMultiplier) || 3.5;
      window.__cbTranslateTipMenuSize = tipMenuMap.size;
      U.log("Content", `Loaded tip menu with ${tipMenuMap.size} entries, font size: ${tipModalFontSize}px, attention rate: $${attentionRate}/min, streak multiplier: ${streakMultiplier}x`);
    } catch (error) {
      U.logError("Content", "Failed to load tip menu:", error);
    }
  }
  function repositionModals() {
    let offset = 20;
    for (const modalInfo of activeModals) {
      modalInfo.element.style.top = `${offset}px`;
      offset += modalInfo.element.offsetHeight * uiScale + 12;
    }
  }
  function getUsernameColor(userData) {
    if (!userData) return "#69a";
    if (userData.isMod) return "#dc0000";
    if (userData.tippedTonsRecently) return "#804baa";
    if (userData.tippedALotRecently) return "#be6aff";
    if (userData.tippedRecently) return "#1e5cfb";
    if (userData.inFanclub) return "#009900";
    if (userData.hasTokens) return "#69a";
    return "rgba(255, 255, 255, 0.7)";
  }
  function showTipModal(amount, fromUser, description, customDuration = MODAL_DURATION, userData = null) {
    if (modalsPaused) {
      U.log("Content", "Modal skipped (paused):", { amount, fromUser, description });
      return;
    }
    const fontSize = tipModalFontSize;
    const minWidth = Math.max(550, fontSize * 40);
    const padding = Math.max(24, fontSize * 1.6);
    const modalDuration = customDuration;
    const earningsColor = getEarningsColor(sessionEarnings);
    const usernameColor = getUsernameColor(userData);
    const modal = document.createElement("div");
    modal.style.cssText = `
        position: fixed;
        left: 20px;
        top: 20px;
        z-index: 10000;
        background: rgba(17, 24, 39, 0.85);
        backdrop-filter: blur(12px);
        -webkit-backdrop-filter: blur(12px);
        border: 1px solid rgba(255, 255, 255, 0.3);
        border-radius: 8px;
        padding: ${padding}px ${padding * 1.25}px;
        padding-top: ${padding + 6}px;
        min-width: ${minWidth}px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.6);
        opacity: 0;
        transition: opacity 0.3s ease, top 0.15s ease;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Inter', system-ui, sans-serif;
        overflow: hidden;
        transform: scale(${uiScale});
        transform-origin: top left;
    `;
    const progressBar = document.createElement("div");
    progressBar.style.cssText = `
        position: absolute;
        top: 0;
        left: 0;
        height: 4px;
        width: 100%;
        background: ${earningsColor.hex};
        transition: width ${modalDuration}ms linear;
    `;
    modal.appendChild(progressBar);
    const scaledFontSize = fontSize * 2;
    const userFontSize = Math.max(14, fontSize * 1.2);
    const content = document.createElement("div");
    content.style.cssText = `
        display: flex;
        flex-direction: column;
        gap: 6px;
        color: #ffffff;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Inter', system-ui, sans-serif;
    `;
    const modelEarns = (amount * 0.05).toFixed(2);
    const userLine = document.createElement("div");
    userLine.style.cssText = `
        display: flex;
        align-items: center;
        gap: 6px;
        font-size: ${userFontSize}px;
        color: rgba(255, 255, 255, 0.6);
        font-weight: 500;
    `;
    const userSpan = document.createElement("span");
    userSpan.textContent = `@${fromUser}`;
    userSpan.style.color = usernameColor;
    userSpan.style.fontWeight = "600";
    const tippedSpan = document.createElement("span");
    tippedSpan.textContent = "tipped";
    const amountSpan = document.createElement("span");
    amountSpan.textContent = amount;
    amountSpan.style.color = "#ffd700";
    const arrowSpan = document.createElement("span");
    arrowSpan.textContent = "\u2192";
    arrowSpan.style.color = "rgba(255, 255, 255, 0.4)";
    const modelSpan = document.createElement("span");
    modelSpan.textContent = `$${modelEarns}`;
    modelSpan.style.cssText = `
        color: ${earningsColor.hex};
        font-weight: 600;
    `;
    userLine.appendChild(userSpan);
    userLine.appendChild(tippedSpan);
    userLine.appendChild(amountSpan);
    userLine.appendChild(arrowSpan);
    userLine.appendChild(modelSpan);
    const descLine = document.createElement("div");
    descLine.textContent = description;
    descLine.style.cssText = `
        font-size: ${scaledFontSize}px;
        font-weight: 500;
        color: #ffffff;
        line-height: 1.3;
    `;
    content.appendChild(userLine);
    content.appendChild(descLine);
    modal.appendChild(content);
    const closeBtn = document.createElement("button");
    closeBtn.textContent = "\xD7";
    closeBtn.style.cssText = `
        position: absolute;
        top: 12px;
        right: 14px;
        background: none;
        border: none;
        color: rgba(255, 255, 255, 0.6);
        font-size: 28px;
        cursor: pointer;
        padding: 0;
        line-height: 1;
        transition: color 0.2s ease;
    `;
    closeBtn.addEventListener("mouseenter", () => closeBtn.style.color = "#ffffff");
    closeBtn.addEventListener("mouseleave", () => closeBtn.style.color = "rgba(255, 255, 255, 0.6)");
    closeBtn.addEventListener("click", () => removeModal(modalInfo));
    modal.appendChild(closeBtn);
    document.body.appendChild(modal);
    const modalInfo = {
      id: Date.now(),
      element: modal,
      timeout: null
    };
    activeModals.push(modalInfo);
    repositionModals();
    requestAnimationFrame(() => {
      modal.style.opacity = "1";
      requestAnimationFrame(() => {
        progressBar.style.width = "0%";
      });
    });
    modalInfo.timeout = setTimeout(() => {
      removeModal(modalInfo);
    }, modalDuration);
  }
  function removeModal(modalInfo) {
    if (modalInfo.timeout) {
      clearTimeout(modalInfo.timeout);
    }
    const index = activeModals.indexOf(modalInfo);
    if (index > -1) {
      activeModals.splice(index, 1);
    }
    if (modalInfo.element.parentNode) {
      modalInfo.element.remove();
    }
    repositionModals();
  }
  function handleTip(tipData) {
    U.log("Content", "Tip received:", tipData);
    const amount = tipData.amount;
    const fromUser = tipData.from_username || "Unknown";
    const userData = {
      isMod: tipData.is_mod,
      inFanclub: tipData.in_fanclub,
      hasTokens: tipData.has_tokens,
      tippedRecently: tipData.tipped_recently,
      tippedALotRecently: tipData.tipped_alot_recently,
      tippedTonsRecently: tipData.tipped_tons_recently
    };
    addEarnings(amount);
    recordTip(amount);
    if (tipMenuMap.has(amount)) {
      const menuItem = tipMenuMap.get(amount);
      const duration = menuItem.durationOverride || calculateModalDuration(amount);
      showTipModal(amount, fromUser, menuItem.description, duration, userData);
    }
  }
  window.addEventListener("cb-translate-tip", (event) => {
    handleTip(event.detail);
  });
  chrome.storage.onChanged.addListener((changes, area) => {
    if (area === "sync") {
      if (changes.tipMenu) {
        tipMenuMap = parseTipMenu(changes.tipMenu.newValue || "");
        window.__cbTranslateTipMenuSize = tipMenuMap.size;
        U.log("Content", `Tip menu updated: ${tipMenuMap.size} entries`);
      }
      if (changes.tipModalFontSize) {
        tipModalFontSize = parseInt(changes.tipModalFontSize.newValue, 10) || 15;
        U.log("Content", `Tip modal font size updated: ${tipModalFontSize}px`);
      }
      if (changes.attentionRate) {
        attentionRate = parseFloat(changes.attentionRate.newValue) || 3;
        U.log("Content", `Attention rate updated: $${attentionRate}/min`);
      }
      if (changes.streakMultiplier) {
        streakMultiplier = parseFloat(changes.streakMultiplier.newValue) || 3.5;
        U.log("Content", `Streak multiplier updated: ${streakMultiplier}x`);
      }
    }
  });
  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", () => setTimeout(init, C.TIMING.INIT_DELAY));
  } else {
    setTimeout(init, C.TIMING.PAGE_LOAD_CHECK);
  }
})();
