// js/config.js
var Config = {
  // API Configuration
  API_URL: "https://translate-api-70536743886.europe-west4.run.app/translate",
  // Cache Configuration
  CACHE_MAX_SIZE: 200,
  CACHE_STORAGE_KEY: "translationCache",
  // Timing Constants (in milliseconds)
  TIMING: {
    INIT_DELAY: 500,
    PAGE_LOAD_CHECK: 1e3,
    API_TIMEOUT: 5e3,
    STATUS_DISPLAY: 5e3,
    MAX_INIT_ATTEMPTS: 3
  },
  // DOM Selectors
  SELECTORS: {
    // Chat input selectors
    MAIN_CHAT_INPUT: '.theatermodeInputFieldChat[contenteditable="true"]',
    PM_CHAT_INPUT: '.theatermodeInputFieldPm[contenteditable="true"]',
    CHAT_INPUT_FIELD: '.chat-input-field[contenteditable="true"]',
    CHAT_INPUT_TESTID: '[data-testid="chat-input"][contenteditable="true"]',
    // Chat message selectors
    CHAT_MESSAGE: '[data-testid="chat-message"]',
    CHAT_MESSAGE_TEXT: '[data-testid="chat-message-text"]',
    MSG_LIST_WRAPPER: ".msg-list-wrapper-split"
  },
  // Theme Colors
  COLORS: {
    // Backgrounds
    BG_PRIMARY: "rgba(17, 24, 39, 0.95)",
    BG_INPUT: "rgba(255, 255, 255, 0.1)",
    BG_INPUT_FOCUS: "rgba(255, 255, 255, 0.15)",
    BG_BUTTON: "rgba(255, 255, 255, 0.05)",
    BG_BUTTON_HOVER: "rgba(255, 255, 255, 0.15)",
    // Borders
    BORDER_DEFAULT: "rgba(255, 255, 255, 0.3)",
    BORDER_FOCUS: "rgba(255, 255, 255, 0.5)",
    BORDER_BUTTON: "rgba(255, 255, 255, 0.4)",
    BORDER_BUTTON_HOVER: "rgba(255, 255, 255, 0.6)",
    // Text
    TEXT_PRIMARY: "#ffffff",
    TEXT_SECONDARY: "rgba(255, 255, 255, 0.7)",
    TEXT_PLACEHOLDER: "rgba(255, 255, 255, 0.4)",
    // Status Colors
    STATUS_SUCCESS: "rgba(100, 255, 150, 0.9)",
    STATUS_ERROR: "rgba(255, 100, 100, 0.9)",
    STATUS_WARNING: "rgba(255, 200, 100, 0.9)",
    STATUS_INFO: "rgba(255, 255, 255, 0.6)",
    // Translation element colors
    TRANSLATION_BG: "rgb(22, 30, 42)",
    TRANSLATION_TEXT: "rgb(209, 233, 239)",
    TRANSLATION_BORDER: "rgb(79, 145, 162)"
  },
  // Default Language Settings
  DEFAULTS: {
    ROOM_LANGUAGE: "en",
    MY_LANGUAGE: "ru"
  },
  // UI Text
  TEXT: {
    TRANSLATE_BTN: "\u041F\u0435\u0440\u0435\u0432\u0435\u0441\u0442\u0438",
    TRANSLATING: "\u041F\u0435\u0440\u0435\u0432\u043E\u0436\u0443...",
    CONFIGURED: "\u041D\u0430\u0441\u0442\u0440\u043E\u0435\u043D",
    NOT_CONFIGURED: "\u041D\u0435 \u043D\u0430\u0441\u0442\u0440\u043E\u0435\u043D",
    NOT_SET: "\u041D\u0435 \u0443\u0441\u0442\u0430\u043D\u043E\u0432\u043B\u0435\u043D",
    ACTIVE: "\u0410\u043A\u0442\u0438\u0432\u043D\u0430",
    INACTIVE: "\u041D\u0435\u0430\u043A\u0442\u0438\u0432\u043D\u0430",
    UNKNOWN: "\u041D\u0435\u0438\u0437\u0432\u0435\u0441\u0442\u043D\u043E",
    ERROR: "\u041E\u0448\u0438\u0431\u043A\u0430",
    // Error messages
    ERR_ENTER_TEXT: "Please enter some text to translate",
    ERR_CONFIGURE_KEY: "Please configure your API key in extension options",
    ERR_NO_CHAT_INPUT: "Error: Could not find chat input field",
    ERR_API_KEY_REQUIRED: "\u041F\u043E\u0436\u0430\u043B\u0443\u0439\u0441\u0442\u0430, \u0432\u0432\u0435\u0434\u0438\u0442\u0435 API \u043A\u043B\u044E\u0447",
    ERR_INVALID_API_KEY: "Invalid API key. Please check your API key in extension options.",
    ERR_API_FORBIDDEN: "API key is not authorized to use this service.",
    ERR_NO_API_KEY: "API key not configured. Please set it in extension options.",
    ERR_INVALID_RESPONSE: "Invalid API response format",
    // Success messages
    SUCCESS_SETTINGS_SAVED: "\u041D\u0430\u0441\u0442\u0440\u043E\u0439\u043A\u0438 \u0441\u043E\u0445\u0440\u0430\u043D\u0435\u043D\u044B \u0443\u0441\u043F\u0435\u0448\u043D\u043E!",
    SUCCESS_API_KEY_CLEARED: "API \u043A\u043B\u044E\u0447 \u043E\u0447\u0438\u0449\u0435\u043D",
    // Info messages
    INFO_TESTING_KEY: "\u041F\u0440\u043E\u0432\u0435\u0440\u043A\u0430 API \u043A\u043B\u044E\u0447\u0430 \u0438 \u0441\u043E\u0435\u0434\u0438\u043D\u0435\u043D\u0438\u044F...",
    INFO_EXTENSION_READY: "\u0420\u0430\u0441\u0448\u0438\u0440\u0435\u043D\u0438\u0435 \u0433\u043E\u0442\u043E\u0432\u043E! \u0418\u043D\u0442\u0435\u0440\u0444\u0435\u0439\u0441 \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430 \u043F\u043E\u044F\u0432\u0438\u0442\u0441\u044F, \u0435\u0441\u043B\u0438 \u043D\u0430 \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u043E \u043F\u043E\u043B\u0435 \u0432\u0432\u043E\u0434\u0430 \u0447\u0430\u0442\u0430.",
    INFO_CONFIGURE_KEY: '\u041F\u043E\u0436\u0430\u043B\u0443\u0439\u0441\u0442\u0430, \u043D\u0430\u0441\u0442\u0440\u043E\u0439\u0442\u0435 \u0432\u0430\u0448 API \u043A\u043B\u044E\u0447 \u0434\u043B\u044F \u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u044F \u0441\u0435\u0440\u0432\u0438\u0441\u0430 \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430. \u041D\u0430\u0436\u043C\u0438\u0442\u0435 "\u041D\u0430\u0441\u0442\u0440\u043E\u0438\u0442\u044C API \u043A\u043B\u044E\u0447" \u043D\u0438\u0436\u0435.',
    // Placeholders
    PLACEHOLDER_INPUT: "\u0412\u0432\u0435\u0434\u0438\u0442\u0435 \u0440\u0443\u0441\u0441\u043A\u0438\u0439 \u0442\u0435\u043A\u0441\u0442 \u0434\u043B\u044F \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430..."
  }
};

// js/utils.js
var Utils = {
  /**
   * Generate a cache key for translations
   * @param {string} text - Text to translate
   * @param {string} sourceLang - Source language code
   * @param {string} targetLang - Target language code
   * @returns {string}
   */
  generateCacheKey(text, sourceLang, targetLang) {
    return `${sourceLang}:${targetLang}:${text}`;
  },
  /**
   * Initialize or retrieve cache data
   * @param {Object} data - Storage data object
   * @returns {Object} Cache object with translations and order
   */
  initializeCache(data) {
    return data[Config.CACHE_STORAGE_KEY] || { translations: {}, order: [] };
  },
  /**
   * Create a mutation observer for added nodes
   * @param {Function} handler - Function to call for each added element node
   * @param {Object} options - MutationObserver options
   * @returns {MutationObserver}
   */
  createNodeObserver(handler, options = { childList: true, subtree: true }) {
    const observer = new MutationObserver((mutations) => {
      for (const mutation of mutations) {
        if (!mutation.addedNodes.length) continue;
        for (const node of mutation.addedNodes) {
          if (node.nodeType === Node.ELEMENT_NODE) {
            handler(node);
          }
        }
      }
    });
    observer.observe(document.body, options);
    return observer;
  },
  /**
   * Add focus/blur effect handlers to an element
   * @param {HTMLElement} element - Element to add effects to
   * @param {Object} focusStyles - Styles to apply on focus
   * @param {Object} blurStyles - Styles to apply on blur
   */
  addFocusEffects(element, focusStyles, blurStyles) {
    element.addEventListener("focus", () => {
      Object.assign(element.style, focusStyles);
    });
    element.addEventListener("blur", () => {
      Object.assign(element.style, blurStyles);
    });
  },
  /**
   * Add hover effect handlers to an element
   * @param {HTMLElement} element - Element to add effects to
   * @param {Object} hoverStyles - Styles to apply on hover
   * @param {Object} defaultStyles - Styles to apply when not hovering
   */
  addHoverEffects(element, hoverStyles, defaultStyles) {
    element.addEventListener("mouseenter", () => {
      if (!element.disabled) {
        Object.assign(element.style, hoverStyles);
      }
    });
    element.addEventListener("mouseleave", () => {
      if (!element.disabled) {
        Object.assign(element.style, defaultStyles);
      }
    });
  },
  /**
   * Set status message with appropriate styling
   * @param {HTMLElement} element - Status element
   * @param {string} message - Message to display
   * @param {'success'|'error'|'warning'|'info'} type - Status type
   */
  setStatus(element, message, type) {
    const colorMap = {
      success: Config.COLORS.STATUS_SUCCESS,
      error: Config.COLORS.STATUS_ERROR,
      warning: Config.COLORS.STATUS_WARNING,
      info: Config.COLORS.STATUS_INFO
    };
    element.textContent = message;
    element.style.color = colorMap[type] || colorMap.info;
  },
  /**
   * Show a status message that auto-hides (for options/popup pages)
   * @param {HTMLElement} element - Status element
   * @param {string} message - Message to display
   * @param {'success'|'error'|'info'} type - Status type
   * @param {number} duration - Duration in ms before hiding
   */
  showStatusMessage(element, message, type, duration) {
    element.textContent = message;
    element.className = `status ${type} show`;
    setTimeout(() => {
      element.className = "status";
    }, duration || Config.TIMING.STATUS_DISPLAY);
  },
  /**
   * Make a translation API request
   * @param {string} apiUrl - API endpoint URL
   * @param {string} apiKey - API key
   * @param {string} text - Text to translate
   * @param {string} source - Source language code
   * @param {string} target - Target language code
   * @param {number} timeout - Request timeout in ms
   * @returns {Promise<{success: boolean, translatedText?: string, error?: string}>}
   */
  async fetchTranslation(apiUrl, apiKey, text, source, target, timeout) {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), timeout || Config.TIMING.API_TIMEOUT);
    try {
      const response = await fetch(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "X-API-Key": apiKey
        },
        body: JSON.stringify({ text, source, target }),
        signal: controller.signal
      });
      clearTimeout(timeoutId);
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
      }
      const data = await response.json();
      if (data?.data?.translations?.[0]?.translatedText) {
        return {
          success: true,
          translatedText: data.data.translations[0].translatedText
        };
      } else {
        throw new Error("Invalid response format from translation API");
      }
    } catch (error) {
      clearTimeout(timeoutId);
      if (error.name === "AbortError") {
        return { success: false, error: `Connection timeout (${(timeout || Config.TIMING.API_TIMEOUT) / 1e3}s)` };
      } else if (error.message.includes("Failed to fetch") || error.message.includes("NetworkError")) {
        return { success: false, error: "Cannot connect to server - check network connection" };
      } else {
        return { success: false, error: error.message };
      }
    }
  },
  /**
   * Get language settings from storage
   * @returns {Promise<{myLanguage: string, roomLanguage: string}>}
   */
  async getLanguageSettings() {
    const config = await chrome.storage.sync.get(["myLanguage", "roomLanguage"]);
    return {
      myLanguage: config.myLanguage || Config.DEFAULTS.MY_LANGUAGE,
      roomLanguage: config.roomLanguage || Config.DEFAULTS.ROOM_LANGUAGE
    };
  },
  /**
   * Log a message with the extension prefix
   * @param {string} context - Context identifier (e.g., 'BG', 'Content')
   * @param {string} message - Message to log
   * @param {...any} args - Additional arguments
   */
  log(context, message, ...args) {
    console.log(`[Chat Translate ${context}]`, message, ...args);
  },
  /**
   * Log an error with the extension prefix
   * @param {string} context - Context identifier
   * @param {string} message - Error message
   * @param {...any} args - Additional arguments
   */
  logError(context, message, ...args) {
    console.error(`[Chat Translate ${context}]`, message, ...args);
  }
};

// options.js
var C = Config;
var U = Utils;
var form = document.getElementById("optionsForm");
var apiKeyInput = document.getElementById("apiKey");
var roomLanguageSelect = document.getElementById("roomLanguage");
var myLanguageSelect = document.getElementById("myLanguage");
var tipMenuTextarea = document.getElementById("tipMenu");
var attentionRateInput = document.getElementById("attentionRate");
var streakMultiplierInput = document.getElementById("streakMultiplier");
var submitBtn = document.querySelector('button[type="submit"]');
var clearBtn = document.getElementById("clearBtn");
var statusDiv = document.getElementById("status");
var apiKeyStatusSpan = document.getElementById("apiKeyStatus");
function validateTipMenu() {
  const text = tipMenuTextarea.value.trim();
  const tipMenuError = document.getElementById("tipMenuError");
  if (!text) {
    tipMenuTextarea.classList.remove("invalid");
    tipMenuError.textContent = "";
    tipMenuError.style.display = "none";
    return { valid: true, errors: [] };
  }
  const lines = text.split("\n");
  const errors = [];
  lines.forEach((line, index) => {
    const trimmed = line.trim();
    if (!trimmed) return;
    const match = trimmed.match(/^(\d+)(?::|[\s-]+)\s*(.+)$/);
    if (!match) {
      errors.push(`\u0421\u0442\u0440\u043E\u043A\u0430 ${index + 1}: "${trimmed.substring(0, 20)}${trimmed.length > 20 ? "..." : ""}" \u2014 \u043D\u0435\u0432\u0435\u0440\u043D\u044B\u0439 \u0444\u043E\u0440\u043C\u0430\u0442`);
    } else {
      const description = match[2].trim();
      const durationMatch = description.match(/^(.+?)\s*\|\s*(.+)$/);
      if (durationMatch) {
        const duration = durationMatch[2].trim();
        if (!/^\d+$/.test(duration)) {
          errors.push(`\u0421\u0442\u0440\u043E\u043A\u0430 ${index + 1}: \u0432\u0440\u0435\u043C\u044F \u0434\u043E\u043B\u0436\u043D\u043E \u0431\u044B\u0442\u044C \u0447\u0438\u0441\u043B\u043E\u043C (\u0441\u0435\u043A\u0443\u043D\u0434\u044B)`);
        }
      }
    }
  });
  if (errors.length > 0) {
    tipMenuTextarea.classList.add("invalid");
    tipMenuError.innerHTML = errors.slice(0, 3).join("<br>") + (errors.length > 3 ? `<br>...\u0438 \u0435\u0449\u0451 ${errors.length - 3} \u043E\u0448\u0438\u0431\u043E\u043A` : "");
    tipMenuError.style.display = "block";
    return { valid: false, errors };
  }
  tipMenuTextarea.classList.remove("invalid");
  tipMenuError.textContent = "";
  tipMenuError.style.display = "none";
  return { valid: true, errors: [] };
}
function validateForm() {
  const apiKey = apiKeyInput.value.trim();
  const apiKeyValid = apiKey.length > 0;
  const tipMenuResult = validateTipMenu();
  if (apiKeyValid) {
    apiKeyInput.classList.remove("invalid");
  } else {
    apiKeyInput.classList.add("invalid");
  }
  const isValid = apiKeyValid && tipMenuResult.valid;
  submitBtn.disabled = !isValid;
  return isValid;
}
async function loadSettings() {
  try {
    const result = await chrome.storage.sync.get(["apiKey", "roomLanguage", "myLanguage", "tipMenu", "attentionRate", "streakMultiplier"]);
    if (result.apiKey) {
      apiKeyInput.value = result.apiKey;
      updateApiKeyStatus(true);
    } else {
      updateApiKeyStatus(false);
    }
    roomLanguageSelect.value = result.roomLanguage || C.DEFAULTS.ROOM_LANGUAGE;
    myLanguageSelect.value = result.myLanguage || C.DEFAULTS.MY_LANGUAGE;
    tipMenuTextarea.value = result.tipMenu || "";
    attentionRateInput.value = result.attentionRate || 3;
    streakMultiplierInput.value = result.streakMultiplier || 4;
    validateForm();
  } catch (error) {
    showStatus("Error loading settings: " + error.message, "error");
  }
}
function updateApiKeyStatus(isConfigured) {
  if (isConfigured) {
    apiKeyStatusSpan.textContent = C.TEXT.CONFIGURED;
    apiKeyStatusSpan.className = "api-key-status configured";
  } else {
    apiKeyStatusSpan.textContent = C.TEXT.NOT_CONFIGURED;
    apiKeyStatusSpan.className = "api-key-status not-configured";
  }
}
async function testTranslation(apiKey) {
  const result = await U.fetchTranslation(
    C.API_URL,
    apiKey,
    "\u041F\u0440\u0438\u0432\u0435\u0442",
    // Test text
    "ru",
    "en",
    C.TIMING.API_TIMEOUT
  );
  return result;
}
async function saveSettings(e) {
  e.preventDefault();
  if (!validateForm()) {
    showStatus(C.TEXT.ERR_API_KEY_REQUIRED, "error");
    return;
  }
  const apiKey = apiKeyInput.value.trim();
  showStatus(C.TEXT.INFO_TESTING_KEY, "info");
  const testResult = await testTranslation(apiKey);
  if (!testResult.success) {
    showStatus(`\u041F\u0440\u043E\u0432\u0435\u0440\u043A\u0430 \u043D\u0435 \u0443\u0434\u0430\u043B\u0430\u0441\u044C: ${testResult.error}`, "error");
    return;
  }
  try {
    const roomLanguage = roomLanguageSelect.value;
    const myLanguage = myLanguageSelect.value;
    const tipMenu = tipMenuTextarea.value.trim();
    const attentionRate = parseFloat(attentionRateInput.value) || 3;
    const streakMultiplier = parseFloat(streakMultiplierInput.value) || 4;
    await chrome.storage.sync.set({
      apiKey,
      roomLanguage,
      myLanguage,
      tipMenu,
      attentionRate,
      streakMultiplier
    });
    updateApiKeyStatus(true);
    showStatus("\u2713 " + C.TEXT.SUCCESS_SETTINGS_SAVED, "success");
  } catch (error) {
    showStatus("\u041E\u0448\u0438\u0431\u043A\u0430 \u0441\u043E\u0445\u0440\u0430\u043D\u0435\u043D\u0438\u044F \u043D\u0430\u0441\u0442\u0440\u043E\u0435\u043A: " + error.message, "error");
  }
}
async function clearApiKey() {
  if (!confirm("\u0412\u044B \u0443\u0432\u0435\u0440\u0435\u043D\u044B, \u0447\u0442\u043E \u0445\u043E\u0442\u0438\u0442\u0435 \u043E\u0447\u0438\u0441\u0442\u0438\u0442\u044C \u0432\u0430\u0448 API \u043A\u043B\u044E\u0447? \u0412\u0430\u043C \u043D\u0443\u0436\u043D\u043E \u0431\u0443\u0434\u0435\u0442 \u0432\u0432\u0435\u0441\u0442\u0438 \u0435\u0433\u043E \u0441\u043D\u043E\u0432\u0430 \u0434\u043B\u044F \u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u044F \u0441\u0435\u0440\u0432\u0438\u0441\u0430 \u043F\u0435\u0440\u0435\u0432\u043E\u0434\u0430.")) {
    return;
  }
  try {
    await chrome.storage.sync.remove(["apiKey"]);
    apiKeyInput.value = "";
    updateApiKeyStatus(false);
    validateForm();
    showStatus(C.TEXT.SUCCESS_API_KEY_CLEARED, "success");
  } catch (error) {
    showStatus("\u041E\u0448\u0438\u0431\u043A\u0430 \u043E\u0447\u0438\u0441\u0442\u043A\u0438 API \u043A\u043B\u044E\u0447\u0430: " + error.message, "error");
  }
}
function showStatus(message, type) {
  U.showStatusMessage(statusDiv, message, type, C.TIMING.STATUS_DISPLAY);
}
form.addEventListener("submit", saveSettings);
clearBtn.addEventListener("click", clearApiKey);
apiKeyInput.addEventListener("input", validateForm);
tipMenuTextarea.addEventListener("input", validateForm);
loadSettings();
